/* calcKWW.c */
/* by apj(1997) */
/*		This program calculates normalized comprex relaxasion function
	based on the method in following article:
	G.Williams and D. C. Watts,
	"Further Considerations of Non Symmetrical Dielectric Relaxation
	Behaviour arising from a Simple Empirical Decay Function",
	Trans. Faraday Soc. 67(1971) 1323-1335
*/		
/* #define DEBUG */ 
		
#include <math.h>

#ifdef DEBUG
#include <stdio.h>
#endif

void CalcKWW(double logwt, double beta, double *realpart, double *imagpart);
static double sum_real(int n);
static double sum_imag(int n);
static double realpartf(double x);
static double imagpartf(double x);

/* from Numerical Recipes */
double qromb(double (*func)(double), double a, double b);
double gammln(double xx);
void nrerror(char *msg);


static double a=0, b=50;		/* Integration from 0 to 50 */
static int n = 50;				/* number of terms */
double wt, beta;				

/* return complex KWW function */
void CalcKWW(double logwt, double betaval, double *realpart, double *imagpart)
{
#ifdef DEBUG
	char temp[256];
	
	sprintf(temp, "logwt = %le, beta = %le\015", logwt, betaval);
	XOPNotice(temp);
	
#endif
	/* get parameters */
	wt = pow(10, logwt);		/* base of 'log' used in the reference is 10 */ 
	beta = betaval;

	if(0.0 < beta && beta <= 0.2){
		*realpart = sum_real(n);
		*imagpart = sum_imag(n);
	}
	else if(0.2 < beta && beta <= 0.3){
		if(logwt < -2){
			*realpart = qromb(realpartf, a, b);	
			*imagpart = qromb(imagpartf, a, b);
		}
		else{
			*realpart = sum_real(n);
			*imagpart = sum_imag(n);
		}
	}
	else if(0.3 < beta && beta <= 0.4){
		if(logwt < -1){
			*realpart = qromb(realpartf, a, b);	
			*imagpart = qromb(imagpartf, a, b);
		}
		else{
			*realpart = sum_real(n);
			*imagpart = sum_imag(n);
		}
	}
	else if(0.4 < beta && beta <= 0.5){
		if(logwt < 0){
			*realpart = qromb(realpartf, a, b);	
			*imagpart = qromb(imagpartf, a, b);
		}
		else{
			*realpart = sum_real(n);
			*imagpart = sum_imag(n);
		}
	}
	else if(0.5 < beta && beta <= 1.0){
		if(logwt < 1.0){
			*realpart = qromb(realpartf, a, b);	
			*imagpart = qromb(imagpartf, a, b);
		}
		else{
			*realpart = sum_real(n);
			*imagpart = sum_imag(n);
		}
	}
	else{
		*realpart = 1.0e6;
		*imagpart = 1.0e6;
	}
	
#ifdef DEBUG
	sprintf(temp, "real = %le, imag = %le\015", *realpart, *imagpart);
	XOPNotice(temp);
#endif

}


#define M_PI        3.14159265358979323846
/* calculate series : eq(5) */
static double sum_real(int num)
{
	int i;
	static double sum;
	
	sum = 0;
	for(i = num ; i >= 1 ; i--){
		sum += pow((double)-1, (double)(i-1))  
		       / pow(wt, (double)i*beta) 
		       * exp(gammln((double)i*beta+(double)1) - gammln((double)(i+1)))
		       * cos((double)i * beta * M_PI / (double)2);
	}
	
	return sum;
}

static double sum_imag(int num)
{
	int i;
	static double sum;
	
	sum = 0;
	for(i = num ; i >= 1 ; i--){
		sum += pow((double)-1, (double)(i-1)) 
		       / pow(wt, (double)i*beta) 
		       * exp(gammln((double)i*beta+(double)1) - gammln((double)(i+1)))
		       * (-sin((double)i * beta * M_PI / (double)2));
	}
	
	return sum;
}


#undef M_PI

/* real and imaginary part of integration: eq.(7)*/
/* x: variable */
/* wt: omega*tau */
/* beta:relaxation parameter */
static double realpartf(double x)
{
	return exp(-x) * cos(-wt*pow(x, (double)1/beta));
}

static double imagpartf(double x)
{
	return exp(-x) * sin(-wt*pow(x, (double)1/beta));
}


void nrerror(char *msg)
{
	/* void */
}
